/*
 |--------------------------------------------------------------------------
 | This file contains examples of how to use this plugin
 |--------------------------------------------------------------------------
 |
 | To see what the pdfs generated by these examples looks like you can open
 | ´examples.html´ or go to http://simonbengtsson.github.io/jsPDF-AutoTable.
 |
 | To make it possible to view each example in examples.html some extra code
 | are added to the examples below. For example they return their jspdf
 | doc instance and gets generated data from the library faker.js. However you
 | can of course use this plugin how you wish and the simplest first example
 | below would look like this without any extras:
 |
 | var columns = ["ID", "Name", "Age", "City"];
 |
 | var data = [
 |     [1, "Jonatan", 25, "Gothenburg"],
 |     [2, "Simon", 23, "Gothenburg"],
 |     [3, "Hanna", 21, "Stockholm"]
 | ];
 |
 | var doc = new jsPDF('p', 'pt');
 | doc.autoTable(columns, data);
 | doc.save("table.pdf");
 |
 */

var examples = {};

// Default - shows what a default table looks like
examples.auto = function () {
    var doc = new jsPDF('p', 'pt');
    doc.autoTable(getColumns(), getData());
    return doc;
};

// Minimal - shows how compact tables can be drawn
examples.minimal = function () {
    var doc = new jsPDF('p', 'pt');
    doc.autoTable(getColumns(), getData(), {
        tableWidth: 'wrap',
        styles: {cellPadding: 2},
        headerStyles: {rowHeight: 15, fontSize: 8},
        bodyStyles: {rowHeight: 12, fontSize: 8, valign: 'middle'}
    });
    return doc;
};

// Long data - shows how the overflow features looks and can be used
examples.long = function () {
    var doc = new jsPDF('l', 'pt');
    var columnsLong = getColumns().concat([
        {title: shuffleSentence(), dataKey: "text"},
        {title: "Text with a\nlinebreak", dataKey: "text2"}
    ]);

    doc.text("Overflow 'ellipsize' (default)", 10, 40);
    doc.autoTable(columnsLong, getData(), {
        startY: 55,
        margin: {horizontal: 10},
        columnStyles: {text: {columnWidth: 250}}
    });

    doc.text("Overflow 'hidden'", 10, doc.autoTableEndPosY() + 30);
    doc.autoTable(columnsLong, getData(), {
        startY: doc.autoTableEndPosY() + 45,
        margin: {horizontal: 10},
        styles: {overflow: 'hidden'},
        columnStyles: {email: {columnWidth: 160}}
    });

    doc.text("Overflow 'linebreak'", 10, doc.autoTableEndPosY() + 30);
    doc.autoTable(columnsLong, getData(3), {
        startY: doc.autoTableEndPosY() + 45,
        margin: {horizontal: 10},
        styles: {overflow: 'linebreak'},
        bodyStyles: {valign: 'top'},
        columnStyles: {email: {columnWidth: 'wrap'}},
    });

    return doc;
};

// Content - shows how tables can be integrated with any other pdf content
examples.content = function () {
    var doc = new jsPDF({unit: 'pt', lineHeight: 1.5, orientation: 'p'});

    doc.setFontSize(18);
    doc.text('A story about Miusov', 40, 60);
    doc.setFontSize(11);
    doc.setTextColor(100);
    var text = doc.splitTextToSize(shuffleSentence(faker.lorem.words(55)) + '.', doc.internal.pageSize.width - 80, {});
    doc.text(text, 40, 80);

    var cols = getColumns();
    cols.splice(0, 2);
    doc.autoTable(cols, getData(40), {startY: 150});

    doc.text(text, 40, doc.autoTableEndPosY() + 30);

    return doc;
};

// Multiple - shows how multiple tables can be drawn both horizontally and vertically
examples.multiple = function () {
    var doc = new jsPDF('p', 'pt');
    doc.setFontSize(22);
    doc.text("Multiple tables", 40, 60);
    doc.setFontSize(12);

    doc.autoTable(getColumns().slice(0, 3), getData(), {
        startY: 90,
        pageBreak: 'avoid',
        margin: {right: 305}
    });

    doc.autoTable(getColumns().slice(0, 3), getData(), {
        startY: 90,
        pageBreak: 'avoid',
        margin: {left: 305}
    });

    for (var j = 0; j < 6; j++) {
        doc.autoTable(getColumns(), getData(9), {
            startY: doc.autoTableEndPosY() + 30,
            pageBreak: 'avoid',
        });
    }

    return doc;
};

// From html - shows how pdf tables can be be drawn from html tables
examples.html = function () {
    var doc = new jsPDF('p', 'pt');
    doc.text("From HTML", 40, 50);
    var res = doc.autoTableHtmlToJson(document.getElementById("basic-table"));
    doc.autoTable(res.columns, res.data, {startY: 60});
    return doc;
};

// Header and footers - shows how header and footers can be drawn
examples['header-footer'] = function () {
    var doc = new jsPDF('p', 'pt');

    var header = function (data) {
        doc.setFontSize(20);
        doc.setTextColor(40);
        doc.setFontStyle('normal');
        doc.addImage(headerImgData, 'JPEG', data.settings.margin.left, 40, 25, 25);
        doc.text("Report", data.settings.margin.left + 35, 60);
    };

    var totalPagesExp = "{total_pages_count_string}";
    var footer = function (data) {
        var str = "Page " + data.pageCount;
        // Total page number plugin only available in jspdf v1.0+
        if (typeof doc.putTotalPages === 'function') {
            str = str + " of " + totalPagesExp;
        }
        doc.text(str, data.settings.margin.left, doc.internal.pageSize.height - 30);
    };

    var options = {
        beforePageContent: header,
        afterPageContent: footer,
        margin: {top: 80}
    };
    doc.autoTable(getColumns(), getData(40), options);

    // Total page number plugin only available in jspdf v1.0+
    if (typeof doc.putTotalPages === 'function') {
        doc.putTotalPages(totalPagesExp);
    }

    return doc;
};

// Themes - shows how the different themes looks
examples.themes = function () {
    var doc = new jsPDF('p', 'pt');
    doc.setFontSize(12);
    doc.setFontStyle('bold');

    doc.text('Theme "striped"', 40, 50);
    doc.autoTable(getColumns(), getData(), {startY: 60});

    doc.text('Theme "grid"', 40, doc.autoTableEndPosY() + 30);
    doc.autoTable(getColumns(), getData(), {startY: doc.autoTableEndPosY() + 40, theme: 'grid'});

    doc.text('Theme "plain"', 40, doc.autoTableEndPosY() + 30);
    doc.autoTable(getColumns(), getData(), {startY: doc.autoTableEndPosY() + 40, theme: 'plain'});

    return doc;
};

// Horizontal - shows how tables can be drawn with horizontal headers
examples.horizontal = function () {
    var doc = new jsPDF('p', 'pt');
    doc.autoTable(getColumns().splice(1,4), getData(), {
        drawHeaderRow: function() {
            // Don't draw header row
            return false;
        },
        columnStyles: {
            first_name: {fillColor: [41, 128, 185], textColor: 255, fontStyle: 'bold'}
        }
    });
    return doc;
};


// Custom style - shows how custom styles can be applied to tables
examples.custom = function () {
    var doc = new jsPDF('p', 'pt');
    doc.autoTable(getColumns().slice(2, 6), getData(20), {
        styles: {
            font: 'courier',
            fillStyle: 'DF',
            lineColor: [44, 62, 80],
            lineWidth: 2
        },
        headerStyles: {
            fillColor: [44, 62, 80],
            fontSize: 15,
            rowHeight: 30
        },
        bodyStyles: {
            fillColor: [52, 73, 94],
            textColor: 240
        },
        alternateRowStyles: {
            fillColor: [74, 96, 117]
        },
        columnStyles: {
            email: {
                fontStyle: 'bold'
            }
        },
        createdCell: function (cell, data) {
            if (data.column.dataKey === 'expenses') {
                cell.styles.halign = 'right';
                if (cell.raw > 600) {
                    cell.styles.textColor = [255, 100, 100];
                    cell.styles.fontStyle = 'bolditalic';
                }
                cell.text = '$' + cell.text;
            } else if (data.column.dataKey === 'country') {
                cell.text = cell.raw.split(' ')[0];
            }
        }
    });
    return doc;
};

// Custom style - shows how custom styles can be applied to tables
examples.spans = function () {
    var doc = new jsPDF('p', 'pt');
    doc.setFontSize(12);
    doc.setTextColor(0);
    doc.setFontStyle('bold');
    doc.text('Col and row span', 40, 50);
    var data = getData(20);
    data.sort(function (a, b) {
        return parseFloat(b.expenses) - parseFloat(a.expenses);
    });
    doc.autoTable(getColumns(), data, {
        theme: 'grid',
        startY: 60,
        drawRow: function (row, data) {
            // Colspan
            doc.setFontStyle('bold');
            doc.setFontSize(10);
            if (row.index === 0) {
                doc.setTextColor(200, 0, 0);
                doc.rect(data.settings.margin.left, row.y, data.table.width, 20, 'S');
                doc.autoTableText("Priority Group", data.settings.margin.left + data.table.width / 2, row.y + row.height / 2, {
                    halign: 'center',
                    valign: 'middle'
                });
                data.cursor.y += 20;
            } else if (row.index === 5) {
                doc.rect(data.settings.margin.left, row.y, data.table.width, 20, 'S');
                doc.autoTableText("Other Groups", data.settings.margin.left + data.table.width / 2, row.y + row.height / 2, {
                    halign: 'center',
                    valign: 'middle'
                });
                data.cursor.y += 20;
            }
        },
        drawCell: function (cell, data) {
            // Rowspan
            if (data.column.dataKey === 'id') {
                if (data.row.index % 5 === 0) {
                    doc.rect(cell.x, cell.y, data.table.width, cell.height * 5, 'S');
                    doc.autoTableText(data.row.index / 5 + 1 + '', cell.x + cell.width / 2, cell.y + cell.height * 5 / 2, {
                        halign: 'center',
                        valign: 'middle'
                    });
                }
                return false;
            }
        }
    });
    return doc;
};

/*
 |--------------------------------------------------------------------------
 | Below is some helper functions for the examples
 |--------------------------------------------------------------------------
 */

// Returns a new array each time to avoid pointer issues
var getColumns = function () {
    return [
        {title: "ID", dataKey: "id"},
        {title: "Name", dataKey: "first_name"},
        {title: "Email", dataKey: "email"},
        {title: "City", dataKey: "city"},
        {title: "Country", dataKey: "country"},
        {title: "Expenses", dataKey: "expenses"}
    ];
};

// Uses the faker.js library to get random data.
function getData(rowCount) {
    rowCount = rowCount || 4;
    var sentence = faker.lorem.words(12);
    var data = [];
    for (var j = 1; j <= rowCount; j++) {
        data.push({
            id: j,
            first_name: faker.name.findName(),
            email: faker.internet.email(),
            country: faker.address.country(),
            city: faker.address.city(),
            expenses: faker.finance.amount(),
            text: shuffleSentence(sentence),
            text2: shuffleSentence(sentence)
        });
    }
    return data;
}

function shuffleSentence(words) {
    words = words || faker.lorem.words(8);
    var str = faker.helpers.shuffle(words).join(' ').trim();
    return str.charAt(0).toUpperCase() + str.slice(1);
}

// Use http://dopiaza.org/tools/datauri or similar service to convert an image into image data
var headerImgData = 'data:image/png;base64,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';